%% MARQUISE post series: Density matrix formalism
% Time evolution of the density matrix
% Quentin Stern - August 26, 2025

clear

%% Time-independent Hamiltonian

omega0  = 2*pi*1;   % Larmor freq (rad.s-1)
dt      = 0.01;     % Time increments (s)
K       = 300;      % Number of increments (-)
S       = 1/2;      % Spin system - single spin 1/2

% Time axis and empty vectors to store results 
% of numerical propatation
t       = dt*(0:K-1);
It      = zeros(3,K);

% Initial state
rho0    = I(S,'x');

% Hamiltonian and propagator during dt
H0      = omega0*I(S,'z');
U0      = expm(-1i*H0*dt);

% Observables
O{1}    = I(S,'x');
O{2}    = I(S,'y');
O{3}    = I(S,'z');

% Initialization of the system state
rho = rho0;

% Loop for propagation
for k = 1:K
    % Expectation values at time (k-1)*dt
    for i = 1:length(O)
        It(i,k) = real(trace(O{i}*rho));
    end

    % Propagation during dt
    rho = U0*rho*U0';
end

% Analytical solution
Ita = [cos(omega0*t)/2; sin(omega0*t)/2; zeros(1,K)];

% Plot of the results
figure(1), lw = 1.5;
plot(t,It,'LineWidth',lw), hold on
plot(t,Ita,'k--','LineWidth',lw), hold off
yline([+1 -1]/2)
legend({'{\itI_x}','{\itI_y}','{\itI_z}'})
xlabel('Time (s)'), ylabel('<{\itI_\mu}> (-)')
yticks((-1:+1)/2), yticklabels({'-1/2','0','+1/2'})
set(gca,'FontSize',12)

%% Time-dependent Hamiltonian

omega0_i= 2*pi*0;   % Larmor freq at the beginning (rad.s-1)
omega0_f= 2*pi*6;   % Larmor freq in the end (rad.s-1)
tmax    = 3;        % Maximum time (s)
K       = 300;      % Number of increments (-)
S       = 1/2;      % Spin system - single spin 1/2

% Time axis and empty vectors to store results 
% of numerical propatation
dt      = tmax/K;
t       = dt*(0:K-1);
It      = zeros(3,K);

% Initial state
rho0    = I(S,'x');

% Observables
O{1}    = I(S,'x');
O{2}    = I(S,'y');
O{3}    = I(S,'z');

% Initialization of the system state
rho = rho0;

% Loop for propagation
for k = 1:K
    % Expectation values at time (k-1)*dt
    for i = 1:length(O)
        It(i,k) = real(trace(O{i}*rho));
    end

    % Hamiltonian and propagator during dt
    omega0  = omega0_i + (omega0_f-omega0_i)*t(k)/max(t);
    H0      = omega0*I(S,'z');
    U0      = expm(-1i*H0*dt);

    % Propagation during dt
    rho = U0*rho*U0';
end

% Plot of the results
figure(1), lw = 1.5;
plot(t,It,'LineWidth',lw)
yline([+1 -1]/2), ylim([-1 1]*0.6)
legend({'{\itI_x}','{\itI_y}','{\itI_z}'})
xlabel('Time (s)'), ylabel('<{\itI_\mu}> (-)')
yticks((-1:+1)/2), yticklabels({'-1/2','0','+1/2'})
set(gca,'FontSize',12)

%% Function

function Itot = I(S,varargin)
    % Quentin Stern 01.03.2024 - Han Lab	
    %
    % Creates an angular momentum operator base on the Pauli matrices
    % Possible inputs are (S, Axis) or (S, k, Axis)
    % 
    % S is a vector contain the spin number of each spin in the system. For
    % example S=1/2 for a single spin and S=[1/2 1] for one spin 1/2 and
    % one spin 1. The dimension of the matrix is sume of prod(2*S+1)
    %
    % mu is 'x', 'y' or 'z', '+', or '-' (Cartesian or ladder operators)
    %
    % k is a vector (or scalar) that specifies which of the spins is
    % represented in the matrix. If k is not defined, I is defined for all
    % spins.
    %
    % For example:
    % I([1/2 1/2 1/2],'z')=I([1/2 1/2 1/2],[1 2 3],'z') = ...
    % I([1/2 1/2 1/2],1,'z')+I([1/2 1/2 1/2],2,'z')+I([1/2 1/2 1/2],3,'z')
    
    % Total number of spins
    N=length(S);

    if size(varargin,2) == 1
        % Example case: I(S,'x')
        Opreration='sum';
        mu=varargin{1};
        k=(1:length(S));
    else
        if strcmpi(varargin{2},'dot')
            % Example case: I(S,[1 2],'dot',3)
            Opreration='dot';
            k1=varargin{1};
            if size(varargin,2) == 3
                k2=varargin{3};
                k=zeros(N);
                for i=1:length(k1)
                    for j=1:length(k2)
                        k(k1(i),k2(j))=1;
                        k(k2(j),k1(i))=1;
                    end
                end
            else
                k=zeros(N);
                disp('WARNING: DOT does not support this number of arguments.')
            end
        else 
            if size(varargin,2) == 2
                % Example case: I(S,[1 2],'x')
                Opreration='sum';
                k=varargin{1};
                mu=varargin{2};
            else
                % Example case: I(S,[1 2],'x',[3 4],'z')
                Opreration='prod';
                K=length(varargin)/2;
                NumberOfOperators=1;
                for i=1:K
                    NumberOfOperators = NumberOfOperators*length(varargin{2*(i-1)+1});
                end
                k=zeros(K,NumberOfOperators);
                mu=cell(K,1);
                for i=1:K
                    k_=1;
                    for j=1:K
                        if i==j
                            k_=kron(k_,varargin{2*(j-1)+1});
                        else
                            l=length(varargin{2*(j-1)+1});
                            k_=kron(k_,ones(1,l));
                        end
                    end
                    k(i,:)=k_;
                    mu{i}=varargin{2*i};
                end
            end
        end
    end
    if strcmp(Opreration,'sum')
        % List of the spins for which the operator is computed
        K=zeros(N,1);
        for i=1:N
            for j=1:max(size(k))
                if k(j)==i
                    K(i)=1;
                end
            end
        end

        Itot=zeros(prod(2*S+1));
        % Iteration over all spins
        for k=1:N
            % Case where the operator contains spin k
            if K(k)==1
                % Initialization of the operator
                I_=1;
                % Loop to contruct the operator for spin k
                for i=1:N
                    % Case of spin k
                    if k==i
                        I_=kron(I_, SingleSpinI(S(k),mu));
                        % Case of non-spin / identity matrix
                    else
                        I_=kron(I_, eye(2*S(i)+1));
                    end
                end
                Itot=Itot+I_;
            end
        end
    elseif strcmp(Opreration,'prod')
        Itot=zeros(prod(2*S+1));
        % Iteration over all operators
        for i=1:NumberOfOperators
            % Initialization of the operator as the identity
            Ii=eye(prod(2*S+1));
            % Iteration over all spins for a particular operator
            for j=1:K
                Ii = Ii*I(S,k(j,i),mu{j});
            end
            Itot=Itot+Ii;
        end
    elseif strcmp(Opreration,'dot')
        Itot=zeros(prod(2*S+1));
        for i=1:N
            for j=i:N
                if k(i,j)==1
                    Itot=Itot+I(S,i,'x',j,'x')+I(S,i,'y',j,'y')+I(S,i,'z',j,'z');
                end
            end
        end
    end
end

function s=SingleSpinI(S,Axis)
    if S==1/2
        if Axis=='x'
            s=1/2*[0 1;1 0];
        elseif Axis=='y'
            s=1/(2*sqrt(-1))*[0 1;-1 0];
        elseif Axis=='z'
            s=1/2*[1 0;0 -1];
        elseif Axis=='+'
            s=[0 1;0 0];
        elseif Axis=='-'
            s=[0 0;1 0];
        else
            s=eye(2);
            disp(['WARNING: ' Axis ' is not a defined operator type.'])
        end
    else
        a=(1:2*S+1)'*ones(1,2*S+1);
        b=ones(2*S+1,1)*(1:2*S+1);
        delta_abp1=a-b;
        delta_abp1(delta_abp1~=1)=0;
        delta_abm1=b-a;
        delta_abm1(delta_abm1~=1)=0;

        if Axis=='x'
            s=(delta_abp1+delta_abm1)/2.*sqrt((S+1)*(a+b-1)-a.*b);
        elseif Axis=='y'
            s=1i*(delta_abp1-delta_abm1)/2.*sqrt((S+1)*(a+b-1)-a.*b);
        elseif Axis=='z'
            s=eye(2*S+1).*(S+1-b);
        elseif Axis=='+'
            s=delta_abm1.*sqrt((S+1)*(a+b-1)-a.*b);
        elseif Axis=='-'   
            s=delta_abp1.*sqrt((S+1)*(a+b-1)-a.*b);
        else
            s=eye(2*S+1);
            disp(['WARNING: ' Axis ' is not a defined operator type.'])
        end
    end
end